/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.token.digest;

import eu.europa.esig.dss.model.DSSException;
import java.nio.ByteBuffer;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DigestInfoEncoder {
    private static final Logger LOG = LoggerFactory.getLogger(DigestInfoEncoder.class);

    private DigestInfoEncoder() {
    }

    public static byte[] encode(String algorithmOid, byte[] digest) {
        Objects.requireNonNull(algorithmOid, "Digest algorithm OID cannot be null!");
        Objects.requireNonNull(digest, "Digest cannot be null!");
        try {
            byte[] oidBytes = DigestInfoEncoder.encodeOid(algorithmOid);
            int oidLength = 2 + oidBytes.length;
            int algorithmIdentifierLength = oidLength + 2;
            int digestLength = 2 + digest.length;
            int totalLength = 2 + algorithmIdentifierLength + 2 + digestLength;
            ByteBuffer buffer = ByteBuffer.allocate(totalLength);
            buffer.put((byte)48);
            buffer.put((byte)(totalLength - 2));
            buffer.put((byte)48);
            buffer.put((byte)algorithmIdentifierLength);
            buffer.put((byte)6);
            buffer.put((byte)oidBytes.length);
            buffer.put(oidBytes);
            buffer.put((byte)5);
            buffer.put((byte)0);
            buffer.put((byte)4);
            buffer.put((byte)digest.length);
            buffer.put(digest);
            return buffer.array();
        }
        catch (Exception e) {
            throw new DSSException(String.format("An error occurred on DigestInfo encoding : %s", e.getMessage()), e);
        }
    }

    private static byte[] encodeOid(String oid) {
        String[] parts = oid.split("\\.");
        if (parts.length < 2) {
            throw new IllegalArgumentException(String.format("The given string '%s' does not represent a valid OID! OID have two or more parts separated by a dot.", oid));
        }
        int byteLength = -1;
        for (String part : parts) {
            byteLength += DigestInfoEncoder.getBytes(Integer.parseInt(part), 1);
        }
        byte[] encodedOid = new byte[byteLength];
        encodedOid[0] = (byte)(40 * Integer.parseInt(parts[0]) + Integer.parseInt(parts[1]));
        int position = 1;
        for (int i = 2; i < parts.length; ++i) {
            position += DigestInfoEncoder.encodeOidPart(encodedOid, Integer.parseInt(parts[i]), position);
        }
        return encodedOid;
    }

    private static int getBytes(int value, int i) {
        if (value < 128) {
            return i;
        }
        return DigestInfoEncoder.getBytes(value >> 7, ++i);
    }

    private static int encodeOidPart(byte[] encodedOid, int value, int position) {
        if (value < 128) {
            encodedOid[position] = (byte)value;
            return 1;
        }
        int byteCount = (int)Math.ceil(Math.log(value) / Math.log(128.0));
        for (int i = byteCount - 1; i >= 0; --i) {
            int encodedByte = value & 0x7F;
            value >>= 7;
            if (i < byteCount - 1) {
                encodedByte |= 0xFFFFFF80;
            }
            DigestInfoEncoder.encodeOidPart(encodedOid, encodedByte, position + i);
        }
        return byteCount;
    }

    public static boolean isEncoded(byte[] data) {
        try {
            int digestLength;
            int oidLength;
            int seqLength;
            if (data == null || data.length < 5) {
                return false;
            }
            int index = 0;
            if (data[index++] != 48) {
                return false;
            }
            if ((seqLength = data[index++] & 0xFF) != data.length - 2) {
                return false;
            }
            if (data[index++] != 48) {
                return false;
            }
            int algorithmIdentifierLength = data[index++] & 0xFF;
            int algorithmIdentifierEnd = index + algorithmIdentifierLength;
            if (data[index++] != 6) {
                return false;
            }
            if (index + (oidLength = data[index++] & 0xFF) > algorithmIdentifierEnd) {
                return false;
            }
            if ((index += oidLength) < algorithmIdentifierEnd) {
                if (data[index] == 5 && index + 1 < algorithmIdentifierEnd && data[index + 1] == 0) {
                    index += 2;
                } else {
                    return false;
                }
            }
            if (index != algorithmIdentifierEnd) {
                return false;
            }
            if (index >= data.length || data[index++] != 4) {
                return false;
            }
            return index + (digestLength = data[index++] & 0xFF) == data.length;
        }
        catch (Exception e) {
            LOG.debug("An error occurred on DigestInfo reading : {}", (Object)e.getMessage(), (Object)e);
            return false;
        }
    }
}

