/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.spi;

import eu.europa.esig.dss.enumerations.QCType;
import eu.europa.esig.dss.enumerations.RoleOfPspOid;
import eu.europa.esig.dss.enumerations.SemanticsIdentifier;
import eu.europa.esig.dss.model.x509.CertificateToken;
import eu.europa.esig.dss.model.x509.extension.PSD2QcType;
import eu.europa.esig.dss.model.x509.extension.PdsLocation;
import eu.europa.esig.dss.model.x509.extension.QCLimitValue;
import eu.europa.esig.dss.model.x509.extension.QcStatements;
import eu.europa.esig.dss.model.x509.extension.RoleOfPSP;
import eu.europa.esig.dss.spi.DSSASN1Utils;
import eu.europa.esig.dss.spi.OID;
import eu.europa.esig.dss.utils.Utils;
import java.util.ArrayList;
import java.util.List;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.qualified.ETSIQCObjectIdentifiers;
import org.bouncycastle.asn1.x509.qualified.MonetaryValue;
import org.bouncycastle.asn1.x509.qualified.QCStatement;
import org.bouncycastle.asn1.x509.qualified.RFC3739QCObjectIdentifiers;
import org.bouncycastle.asn1.x509.qualified.SemanticsInformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class QcStatementUtils {
    private static final Logger LOG = LoggerFactory.getLogger(QcStatementUtils.class);

    private QcStatementUtils() {
    }

    public static QcStatements getQcStatements(CertificateToken certToken) {
        byte[] qcStatementsExtension = certToken.getCertificate().getExtensionValue(Extension.qCStatements.getId());
        if (Utils.isArrayNotEmpty(qcStatementsExtension)) {
            try {
                ASN1Sequence qcStatementsSeq = DSSASN1Utils.getAsn1SequenceFromDerOctetString(qcStatementsExtension);
                QcStatements qcStatements = QcStatementUtils.getQcStatements(qcStatementsSeq);
                qcStatements.checkCritical(certToken);
                return qcStatements;
            }
            catch (Exception e) {
                if (LOG.isDebugEnabled()) {
                    LOG.warn("Unable to extract QcStatements : {}. Obtained binaries : '{}'", (Object)e.getMessage(), (Object)Utils.toBase64(qcStatementsExtension));
                }
                LOG.warn("Unable to extract QcStatements : {}", (Object)e.getMessage());
            }
        }
        return null;
    }

    public static QcStatements getQcStatements(ASN1Sequence qcStatementsSeq) {
        if (qcStatementsSeq == null) {
            return null;
        }
        QcStatements result = new QcStatements();
        result.setOctets(DSSASN1Utils.getDEREncoded(qcStatementsSeq));
        for (int i = 0; i < qcStatementsSeq.size(); ++i) {
            QCStatement statement = QcStatementUtils.getQCStatement(qcStatementsSeq.getObjectAt(i));
            if (statement == null) continue;
            ASN1ObjectIdentifier objectIdentifier = statement.getStatementId();
            String oid = objectIdentifier.getId();
            ASN1Encodable statementInfo = statement.getStatementInfo();
            if (QcStatementUtils.isQcCompliance(oid)) {
                result.setQcCompliance(true);
                continue;
            }
            if (QcStatementUtils.isQcLimitValue(oid)) {
                result.setQcLimitValue(QcStatementUtils.getQcLimitValue(statementInfo));
                continue;
            }
            if (QcStatementUtils.isQcRetentionPeriod(oid)) {
                result.setQcEuRetentionPeriod(QcStatementUtils.getQcEuRetentionPeriod(statementInfo));
                continue;
            }
            if (QcStatementUtils.isQcSSCD(oid)) {
                result.setQcQSCD(true);
                continue;
            }
            if (QcStatementUtils.isQcPds(oid)) {
                result.setQcEuPDS(QcStatementUtils.getQcEuPDS(statementInfo));
                continue;
            }
            if (QcStatementUtils.isQcType(oid)) {
                result.setQcTypes(QcStatementUtils.getQcTypes(statementInfo));
                continue;
            }
            if (QcStatementUtils.isQcCClegislation(oid)) {
                result.setQcLegislationCountryCodes(QcStatementUtils.getQcLegislationCountryCodes(statementInfo));
                continue;
            }
            if (QcStatementUtils.isQcSemanticsIdentifier(oid)) {
                result.setQcSemanticsIdentifier(QcStatementUtils.getQcSemanticsIdentifier(statementInfo));
                continue;
            }
            if (QcStatementUtils.isPsd2QcType(oid)) {
                result.setPsd2QcType(QcStatementUtils.getPsd2QcType(statementInfo));
                continue;
            }
            LOG.warn("Not supported QcStatement with OID : '{}'", (Object)oid);
            result.addOtherOid(oid);
        }
        return result;
    }

    public static boolean isQcCompliance(String oid) {
        return ETSIQCObjectIdentifiers.id_etsi_qcs_QcCompliance.getId().equals(oid);
    }

    public static boolean isQcLimitValue(String oid) {
        return ETSIQCObjectIdentifiers.id_etsi_qcs_LimiteValue.getId().equals(oid);
    }

    public static boolean isQcRetentionPeriod(String oid) {
        return ETSIQCObjectIdentifiers.id_etsi_qcs_RetentionPeriod.getId().equals(oid);
    }

    public static boolean isQcSSCD(String oid) {
        return ETSIQCObjectIdentifiers.id_etsi_qcs_QcSSCD.getId().equals(oid);
    }

    public static boolean isQcPds(String oid) {
        return ETSIQCObjectIdentifiers.id_etsi_qcs_QcPds.getId().equals(oid);
    }

    public static boolean isQcType(String oid) {
        return ETSIQCObjectIdentifiers.id_etsi_qcs_QcType.getId().equals(oid);
    }

    public static boolean isQcCClegislation(String oid) {
        return OID.id_etsi_qcs_QcCClegislation.getId().equals(oid);
    }

    public static boolean isQcSemanticsIdentifier(String oid) {
        return RFC3739QCObjectIdentifiers.id_qcs_pkixQCSyntax_v2.getId().equals(oid);
    }

    public static boolean isPsd2QcType(String oid) {
        return OID.psd2_qcStatement.getId().equals(oid);
    }

    private static QCStatement getQCStatement(ASN1Encodable qcStatement) {
        if (qcStatement != null) {
            try {
                return QCStatement.getInstance(qcStatement);
            }
            catch (Exception e) {
                if (LOG.isDebugEnabled()) {
                    LOG.warn("Unable to extract QCStatement : {}. Obtained sequence binaries : '{}'", (Object)e.getMessage(), (Object)Utils.toBase64(DSSASN1Utils.getDEREncoded(qcStatement)));
                }
                LOG.warn("Unable to extract QCStatement : {}", (Object)e.getMessage());
            }
        }
        return null;
    }

    private static QCLimitValue getQcLimitValue(ASN1Encodable statementInfo) {
        try {
            MonetaryValue monetaryValue = MonetaryValue.getInstance(statementInfo);
            QCLimitValue result = new QCLimitValue();
            result.setCurrency(monetaryValue.getCurrency().getAlphabetic());
            result.setAmount(monetaryValue.getAmount().intValue());
            result.setExponent(monetaryValue.getExponent().intValue());
            return result;
        }
        catch (Exception e) {
            if (LOG.isDebugEnabled()) {
                LOG.warn("Unable to extract QcLimitValue : {}. Obtained binaries : '{}'", (Object)e.getMessage(), (Object)Utils.toBase64(DSSASN1Utils.getDEREncoded(statementInfo)));
            } else {
                LOG.warn("Unable to extract QcLimitValue : {}", (Object)e.getMessage());
            }
            return null;
        }
    }

    private static Integer getQcEuRetentionPeriod(ASN1Encodable statementInfo) {
        try {
            ASN1Integer integer = ASN1Integer.getInstance(statementInfo);
            return integer.intValueExact();
        }
        catch (Exception e) {
            if (LOG.isDebugEnabled()) {
                LOG.warn("Unable to extract QcEuRetentionPeriod : {}. Obtained binaries : '{}'", (Object)e.getMessage(), (Object)Utils.toBase64(DSSASN1Utils.getDEREncoded(statementInfo)));
            } else {
                LOG.warn("Unable to extract QcEuRetentionPeriod : {}", (Object)e.getMessage());
            }
            return null;
        }
    }

    private static List<PdsLocation> getQcEuPDS(ASN1Encodable statementInfo) {
        ArrayList<PdsLocation> result = new ArrayList<PdsLocation>();
        try {
            ASN1Sequence sequence = ASN1Sequence.getInstance(statementInfo);
            for (int i = 0; i < sequence.size(); ++i) {
                ASN1Encodable e1 = sequence.getObjectAt(i);
                if (e1 instanceof ASN1Sequence) {
                    ASN1Sequence seq = (ASN1Sequence)e1;
                    result.add(QcStatementUtils.getPdsLocation(seq));
                    continue;
                }
                LOG.warn("ASN1Sequence in QcEuPDS does not contain ASN1Sequence, but {}", (Object)e1.getClass().getName());
            }
        }
        catch (Exception e) {
            if (LOG.isDebugEnabled()) {
                LOG.warn("Unable to extract QcEuPDS : {}. Obtained binaries : '{}'", (Object)e.getMessage(), (Object)Utils.toBase64(DSSASN1Utils.getDEREncoded(statementInfo)));
            }
            LOG.warn("Unable to extract QcEuPDS : {}", (Object)e.getMessage());
        }
        return result;
    }

    private static PdsLocation getPdsLocation(ASN1Sequence seq) {
        PdsLocation pdsLocation = new PdsLocation();
        pdsLocation.setUrl(DSSASN1Utils.getString(seq.getObjectAt(0)));
        pdsLocation.setLanguage(DSSASN1Utils.getString(seq.getObjectAt(1)));
        return pdsLocation;
    }

    private static List<QCType> getQcTypes(ASN1Encodable statementInfo) {
        ArrayList<String> oids = new ArrayList<String>();
        try {
            ASN1Sequence sequence = ASN1Sequence.getInstance(statementInfo);
            for (int i = 0; i < sequence.size(); ++i) {
                ASN1Encodable e1 = sequence.getObjectAt(i);
                if (e1 instanceof ASN1ObjectIdentifier) {
                    ASN1ObjectIdentifier oid = (ASN1ObjectIdentifier)e1;
                    oids.add(oid.getId());
                    continue;
                }
                LOG.warn("ASN1Sequence in QcTypes does not contain ASN1ObjectIdentifier, but {}", (Object)e1.getClass().getName());
            }
        }
        catch (Exception e) {
            if (LOG.isDebugEnabled()) {
                LOG.warn("Unable to extract QcTypes : {}. Obtained binaries : '{}'", (Object)e.getMessage(), (Object)Utils.toBase64(DSSASN1Utils.getDEREncoded(statementInfo)));
            }
            LOG.warn("Unable to extract QcTypes : {}", (Object)e.getMessage());
        }
        return QcStatementUtils.getQcTypes(oids);
    }

    public static List<QCType> getQcTypes(List<String> oids) {
        ArrayList<QCType> result = new ArrayList<QCType>();
        for (String oid : oids) {
            if (Utils.isStringNotBlank(oid)) {
                QCType type = QCType.fromOid(oid);
                result.add(type);
                continue;
            }
            LOG.warn("Empty QcType OID is skipped.");
        }
        return result;
    }

    private static List<String> getQcLegislationCountryCodes(ASN1Encodable statementInfo) {
        ArrayList<String> result = new ArrayList<String>();
        try {
            ASN1Sequence sequence = ASN1Sequence.getInstance(statementInfo);
            for (int i = 0; i < sequence.size(); ++i) {
                String countryCode = DSSASN1Utils.getString(sequence.getObjectAt(i));
                if (countryCode == null) continue;
                result.add(countryCode);
            }
        }
        catch (Exception e) {
            if (LOG.isDebugEnabled()) {
                LOG.warn("Unable to extract QcCClegislation : {}. Obtained binaries : '{}'", (Object)e.getMessage(), (Object)Utils.toBase64(DSSASN1Utils.getDEREncoded(statementInfo)));
            }
            LOG.warn("Unable to extract QcCClegislation : {}", (Object)e.getMessage());
        }
        return result;
    }

    private static SemanticsIdentifier getQcSemanticsIdentifier(ASN1Encodable statementInfo) {
        try {
            SemanticsInformation semanticsInfo = SemanticsInformation.getInstance(statementInfo);
            if (semanticsInfo != null && semanticsInfo.getSemanticsIdentifier() != null) {
                return SemanticsIdentifier.fromOid(semanticsInfo.getSemanticsIdentifier().getId());
            }
        }
        catch (Exception e) {
            if (LOG.isDebugEnabled()) {
                LOG.warn("Unable to extract QcSemanticsIdentifiers : {}. Obtained binaries : '{}'", (Object)e.getMessage(), (Object)Utils.toBase64(DSSASN1Utils.getDEREncoded(statementInfo)));
            }
            LOG.warn("Unable to extract QcSemanticsIdentifiers : {}", (Object)e.getMessage());
        }
        return null;
    }

    private static PSD2QcType getPsd2QcType(ASN1Encodable statementInfo) {
        try {
            PSD2QcType result = new PSD2QcType();
            ASN1Sequence sequence = ASN1Sequence.getInstance(statementInfo);
            ASN1Sequence rolesSeq = ASN1Sequence.getInstance(sequence.getObjectAt(0));
            ArrayList<RoleOfPSP> rolesOfPSP = new ArrayList<RoleOfPSP>();
            for (int i = 0; i < rolesSeq.size(); ++i) {
                ASN1Sequence oneRoleSeq = ASN1Sequence.getInstance(rolesSeq.getObjectAt(i));
                RoleOfPSP roleOfPSP = new RoleOfPSP();
                ASN1ObjectIdentifier oid = (ASN1ObjectIdentifier)oneRoleSeq.getObjectAt(0);
                roleOfPSP.setPspOid(RoleOfPspOid.fromOid(oid.getId()));
                roleOfPSP.setPspName(DSSASN1Utils.getString(oneRoleSeq.getObjectAt(1)));
                rolesOfPSP.add(roleOfPSP);
            }
            result.setRolesOfPSP(rolesOfPSP);
            result.setNcaName(DSSASN1Utils.getString(sequence.getObjectAt(1)));
            result.setNcaId(DSSASN1Utils.getString(sequence.getObjectAt(2)));
            return result;
        }
        catch (Exception e) {
            if (LOG.isDebugEnabled()) {
                LOG.warn("Unable to extract PSD2-QcStatement : {}. Obtained binaries : '{}'", (Object)e.getMessage(), (Object)Utils.toBase64(DSSASN1Utils.getDEREncoded(statementInfo)));
            } else {
                LOG.warn("Unable to extract PSD2-QcStatement : {}", (Object)e.getMessage());
            }
            return null;
        }
    }

    public static boolean isQcStatementPresent(QcStatements qcStatements, String qcStatementOid) {
        if (QcStatementUtils.isQcCompliance(qcStatementOid)) {
            return qcStatements.isQcCompliance();
        }
        if (QcStatementUtils.isQcLimitValue(qcStatementOid)) {
            return qcStatements.getQcLimitValue() != null;
        }
        if (QcStatementUtils.isQcRetentionPeriod(qcStatementOid)) {
            return qcStatements.getQcEuRetentionPeriod() != null;
        }
        if (QcStatementUtils.isQcSSCD(qcStatementOid)) {
            return qcStatements.isQcQSCD();
        }
        if (QcStatementUtils.isQcPds(qcStatementOid)) {
            return Utils.isCollectionNotEmpty(qcStatements.getQcEuPDS());
        }
        if (QcStatementUtils.isQcType(qcStatementOid)) {
            return Utils.isCollectionNotEmpty(qcStatements.getQcTypes());
        }
        if (QcStatementUtils.isQcCClegislation(qcStatementOid)) {
            return Utils.isCollectionNotEmpty(qcStatements.getQcLegislationCountryCodes());
        }
        if (QcStatementUtils.isQcSemanticsIdentifier(qcStatementOid)) {
            return qcStatements.getQcSemanticsIdentifier() != null;
        }
        if (QcStatementUtils.isPsd2QcType(qcStatementOid)) {
            return qcStatements.getPsd2QcType() != null;
        }
        return qcStatements.getOtherOids().contains(qcStatementOid);
    }

    public static boolean isQcTypePresent(QcStatements qcStatements, String qcTypeOid) {
        List<QCType> qcTypes = qcStatements.getQcTypes();
        if (Utils.isCollectionNotEmpty(qcTypes)) {
            for (QCType qcType : qcTypes) {
                if (!qcTypeOid.equals(qcType.getOid())) continue;
                return true;
            }
        }
        return false;
    }

    public static boolean isQcLegislationPresent(QcStatements qcStatements, String qcLegislation) {
        List<String> qcLegislationCountryCodes = qcStatements.getQcLegislationCountryCodes();
        if (Utils.isCollectionNotEmpty(qcLegislationCountryCodes)) {
            return qcLegislationCountryCodes.contains(qcLegislation);
        }
        return false;
    }
}

